import tkinter as tk
import subprocess
import re

# Function to get power values
def get_power_values():
    cpu_power = 0.0
    gpu_power_amd = 0.0
    gpu_power_nvidia = 0.0

    # Get sensor data
    try:
        sensors_output = subprocess.check_output(["sensors"], text=True)
    except subprocess.CalledProcessError:
        sensors_output = ""

    # Extract CPU power (sum of SVI2_P_Core and SVI2_P_SoC)
    match_cpu = re.findall(r"SVI2_P_(Core|SoC):\s+([\d.]+)\s+W", sensors_output)
    if match_cpu:
        cpu_power = sum(float(value) for _, value in match_cpu)

    # Extract AMD GPU power (PPT)
    match_amd = re.search(r"amdgpu-pci-\d+\n.*?PPT:\s+([\d.]+)\s+W", sensors_output, re.DOTALL)
    if match_amd:
        gpu_power_amd = float(match_amd.group(1))

    # Extract NVIDIA GPU power (if available)
    try:
        nvidia_output = subprocess.check_output(
            ["nvidia-smi", "--query-gpu=power.draw", "--format=csv,noheader,nounits"],
            text=True
        )
        gpu_power_nvidia = float(nvidia_output.strip())
    except (subprocess.CalledProcessError, FileNotFoundError, ValueError):
        gpu_power_nvidia = 0.0

    # Determine final GPU power (use AMD if no NVIDIA detected)
    gpu_power = gpu_power_nvidia if gpu_power_nvidia > 0 else gpu_power_amd
    total_power = cpu_power + gpu_power

    # Update GUI text
    cpu_label.config(text=f"CPU Power: {cpu_power:.2f} W")
    gpu_label.config(text=f"GPU Power: {gpu_power:.2f} W")
    total_label.config(text=f"Total Power: {total_power:.2f} W")

    # Refresh every 2 seconds
    root.after(2000, get_power_values)

# Create GUI
root = tk.Tk()
root.title("Power Usage Monitor")

cpu_label = tk.Label(root, text="CPU Power: -- W", font=("Arial", 14))
cpu_label.pack(pady=5)

gpu_label = tk.Label(root, text="GPU Power: -- W", font=("Arial", 14))
gpu_label.pack(pady=5)

total_label = tk.Label(root, text="Total Power: -- W", font=("Arial", 14, "bold"))
total_label.pack(pady=10)

# Start auto-refresh
get_power_values()

# Run GUI loop
root.mainloop()
